<?php
// src/MessageProcessor.php

require_once __DIR__ . '/../config.php';

class MessageProcessor {
    public function process($message) {
        $text = $message['text'] ?? '';
        $chatId = $message['chat']['id'];
        $userId = $message['from']['id'];
        $chatType = $message['chat']['type'];
        $isPrivateChat = $chatType === 'private';

        // 1. مدیریت حالت "سیف بات"
        if (stripos($text, BOT_PREFIX) === 0) {
            $query = trim(substr($text, strlen(BOT_PREFIX)));
            if ($query === '') {
                return $this->handleEmptyPrefix($chatId, $isPrivateChat);
            }
            return ['type' => 'ai_query', 'query' => $query, 'chat_id' => $chatId, 'user_id' => $userId];
        }

        // 2. مدیریت دستورات اصلی
        $main_commands = [
            "/start", "/search", "/stats", "/admin", "/listchats",
            "📜 قیمت‌های متنی", "🖼 آخرین تصویر", "🗂 آرشیو تصاویر",
            "🔄 قیمت جدید", "👨‍💼 پشتیبانی", "ℹ️ درباره ربات",
            "🛒 فروشگاه سیف گلد", "🖼 تصویر نرخ", "📜 متن نرخ",
            "متن نرخ", "تصویر نرخ", "🤖 چت با هوش مصنوعی"
        ];

        if (in_array($text, $main_commands)) {
            return ['type' => 'command', 'command' => $text, 'chat_id' => $chatId, 'user_id' => $userId, 'chat_type' => $chatType];
        }

        // 3. مدیریت تاریخ جستجو
        if (preg_match('/^\d{4}[-\/]\d{2}[-\/]\d{2}$/', $text)) {
            return ['type' => 'search_date', 'date' => $text, 'chat_id' => $chatId];
        }

        // 4. مدیریت پیام‌های معمولی در چت خصوصی (برای کمک)
        if ($isPrivateChat) {
            if (in_array(strtolower($text), ['/', '/start', 'راهنما', 'help', 'کمک'])) {
                return ['type' => 'help', 'chat_id' => $chatId];
            }
            return ['type' => 'unknown_private', 'chat_id' => $chatId];
        }

        // 5. پیام‌های گروه یا دیگر نوع چت که با "سیف بات" یا دستورات شروع نشده
        return ['type' => 'ignored'];
    }

    private function handleEmptyPrefix($chatId, $isPrivateChat) {
        $welcomeMessage = $this->getWelcomeMessage($isPrivateChat);
        $keyboard = $this->getMainKeyboard();
        return ['type' => 'welcome', 'chat_id' => $chatId, 'message' => $welcomeMessage, 'keyboard' => $keyboard];
    }

    public function getWelcomeMessage($isPrivateChat) {
        $message = "👋 سلام! من " . BOT_NAME . " هستم.\n\n";
        if ($isPrivateChat) {
            $message .= "برای گفتگو با هوش مصنوعی، پیام خود را با '" . BOT_PREFIX . "' شروع کنید.\n\n";
        }
        $message .= "📝 <b>مثال:</b>\n";
        $message .= "<code>" . BOT_PREFIX . " قیمت دلار چنده؟</code>\n";
        $message .= "<code>" . BOT_PREFIX . " چطور می‌تونم سکه بخرم؟</code>\n\n";
        $message .= "💰 می‌تونید در مورد قیمت‌ها، محصولات و خدمات سوال کنید.\n\n";
        $message .= "• " . BOT_PREFIX . " قیمت طلا چقدر است؟\n";
        $message .= "• " . BOT_PREFIX . " بهترین روش سرمایه گذاری چیست؟\n";
        $message .= "• " . BOT_PREFIX . " نحوه خرید از سیف گلد چطوریه؟";
        return $message;
    }

    public function getMainKeyboard() {
        return [
            'keyboard' => [
                [
                    ['text' => '📜 قیمت‌های متنی'],
                    ['text' => '🖼 آخرین تصویر']
                ],
                [
                    ['text' => '🔄 قیمت جدید'],
                    ['text' => '🗂 آرشیو تصاویر']
                ],
                [
                    ['text' => '🤖 چت با هوش مصنوعی'],
                    ['text' => 'ℹ️ درباره ربات']
                ],
                [
                    ['text' => '👨‍💼 پشتیبانی'],
                    ['text' => '🛒 فروشگاه سیف گلد']
                ]
            ],
            'resize_keyboard' => true,
            'one_time_keyboard' => false,
            'selective' => true
        ];
    }
}